import { __export } from "../../_virtual/rolldown_runtime.mjs";
import { initV2Endpoint } from "../../runtime/manifest.mjs";
import { withInit } from "../../common/onInit.mjs";
import { normalizePath } from "../../common/utilities/path.mjs";
import { wrapTraceContext } from "../trace.mjs";
import { getGlobalOptions, optionsToEndpoint } from "../options.mjs";
import { PathPattern } from "../../common/utilities/path-pattern.mjs";

//#region src/v2/providers/dataconnect.ts
var dataconnect_exports = /* @__PURE__ */ __export({
	mutationExecutedEventType: () => mutationExecutedEventType,
	onMutationExecuted: () => onMutationExecuted
});
/** @internal */
const mutationExecutedEventType = "google.firebase.dataconnect.connector.v1.mutationExecuted";
/**
* Event handler that triggers when a mutation is executed in Firebase Data Connect.
*
* @param mutationOrOpts - Options or string mutation path.
* @param handler - Event handler which is run every time a mutation is executed.
*/
function onMutationExecuted(mutationOrOpts, handler) {
	return onOperation(mutationExecutedEventType, mutationOrOpts, handler);
}
function getOpts(mutationOrOpts) {
	const operationRegex = new RegExp("services/([^/]+)/connectors/([^/]*)/operations/([^/]+)");
	let service;
	let connector;
	let operation;
	let opts;
	if (typeof mutationOrOpts === "string") {
		const path = normalizePath(mutationOrOpts);
		const match = path.match(operationRegex);
		if (!match) {
			throw new Error(`Invalid operation path: ${path}`);
		}
		service = match[1];
		connector = match[2];
		operation = match[3];
		opts = {};
	} else {
		service = mutationOrOpts.service;
		connector = mutationOrOpts.connector;
		operation = mutationOrOpts.operation;
		opts = { ...mutationOrOpts };
		delete opts.service;
		delete opts.connector;
		delete opts.operation;
	}
	return {
		service,
		connector,
		operation,
		opts
	};
}
function makeEndpoint(eventType, opts, service, connector, operation) {
	const baseOpts = optionsToEndpoint(getGlobalOptions());
	const specificOpts = optionsToEndpoint(opts);
	const eventFilters = {};
	const eventFilterPathPatterns = {};
	if (service) {
		if (service.hasWildcards()) {
			eventFilterPathPatterns.service = service.getValue();
		} else {
			eventFilters.service = service.getValue();
		}
	}
	if (connector) {
		if (connector.hasWildcards()) {
			eventFilterPathPatterns.connector = connector.getValue();
		} else {
			eventFilters.connector = connector.getValue();
		}
	}
	if (operation) {
		if (operation.hasWildcards()) {
			eventFilterPathPatterns.operation = operation.getValue();
		} else {
			eventFilters.operation = operation.getValue();
		}
	}
	return {
		...initV2Endpoint(getGlobalOptions(), opts),
		platform: "gcfv2",
		...baseOpts,
		...specificOpts,
		labels: {
			...baseOpts?.labels,
			...specificOpts?.labels
		},
		eventTrigger: {
			eventType,
			eventFilters,
			eventFilterPathPatterns,
			retry: opts.retry ?? false
		}
	};
}
function makeParams(event, service, connector, operation) {
	return {
		...service?.extractMatches(event.service),
		...connector?.extractMatches(event.connector),
		...operation?.extractMatches(event.operation)
	};
}
function onOperation(eventType, mutationOrOpts, handler) {
	const { service, connector, operation, opts } = getOpts(mutationOrOpts);
	const servicePattern = service ? new PathPattern(service) : undefined;
	const connectorPattern = connector ? new PathPattern(connector) : undefined;
	const operationPattern = operation ? new PathPattern(operation) : undefined;
	const func = (raw) => {
		const event = raw;
		const params = makeParams(event, servicePattern, connectorPattern, operationPattern);
		const { authtype, authid, service: service$1, connector: connector$1, operation: operation$1,...rest } = event;
		const dataConnectEvent = {
			...rest,
			authType: authtype,
			authId: authid,
			params
		};
		return wrapTraceContext(withInit(handler))(dataConnectEvent);
	};
	func.run = handler;
	func.__endpoint = makeEndpoint(eventType, opts, servicePattern, connectorPattern, operationPattern);
	return func;
}

//#endregion
export { dataconnect_exports, mutationExecutedEventType, onMutationExecuted };