const require_rolldown_runtime = require('../../_virtual/rolldown_runtime.js');
const require_runtime_manifest = require('../../runtime/manifest.js');
const require_common_onInit = require('../../common/onInit.js');
const require_common_utilities_path = require('../../common/utilities/path.js');
const require_v2_trace = require('../trace.js');
const require_v2_options = require('../options.js');
const require_common_utilities_path_pattern = require('../../common/utilities/path-pattern.js');

//#region src/v2/providers/dataconnect.ts
var dataconnect_exports = /* @__PURE__ */ require_rolldown_runtime.__export({
	mutationExecutedEventType: () => mutationExecutedEventType,
	onMutationExecuted: () => onMutationExecuted
});
/** @internal */
const mutationExecutedEventType = "google.firebase.dataconnect.connector.v1.mutationExecuted";
/**
* Event handler that triggers when a mutation is executed in Firebase Data Connect.
*
* @param mutationOrOpts - Options or string mutation path.
* @param handler - Event handler which is run every time a mutation is executed.
*/
function onMutationExecuted(mutationOrOpts, handler) {
	return onOperation(mutationExecutedEventType, mutationOrOpts, handler);
}
function getOpts(mutationOrOpts) {
	const operationRegex = new RegExp("services/([^/]+)/connectors/([^/]*)/operations/([^/]+)");
	let service;
	let connector;
	let operation;
	let opts;
	if (typeof mutationOrOpts === "string") {
		const path = require_common_utilities_path.normalizePath(mutationOrOpts);
		const match = path.match(operationRegex);
		if (!match) {
			throw new Error(`Invalid operation path: ${path}`);
		}
		service = match[1];
		connector = match[2];
		operation = match[3];
		opts = {};
	} else {
		service = mutationOrOpts.service;
		connector = mutationOrOpts.connector;
		operation = mutationOrOpts.operation;
		opts = { ...mutationOrOpts };
		delete opts.service;
		delete opts.connector;
		delete opts.operation;
	}
	return {
		service,
		connector,
		operation,
		opts
	};
}
function makeEndpoint(eventType, opts, service, connector, operation) {
	const baseOpts = require_v2_options.optionsToEndpoint(require_v2_options.getGlobalOptions());
	const specificOpts = require_v2_options.optionsToEndpoint(opts);
	const eventFilters = {};
	const eventFilterPathPatterns = {};
	if (service) {
		if (service.hasWildcards()) {
			eventFilterPathPatterns.service = service.getValue();
		} else {
			eventFilters.service = service.getValue();
		}
	}
	if (connector) {
		if (connector.hasWildcards()) {
			eventFilterPathPatterns.connector = connector.getValue();
		} else {
			eventFilters.connector = connector.getValue();
		}
	}
	if (operation) {
		if (operation.hasWildcards()) {
			eventFilterPathPatterns.operation = operation.getValue();
		} else {
			eventFilters.operation = operation.getValue();
		}
	}
	return {
		...require_runtime_manifest.initV2Endpoint(require_v2_options.getGlobalOptions(), opts),
		platform: "gcfv2",
		...baseOpts,
		...specificOpts,
		labels: {
			...baseOpts?.labels,
			...specificOpts?.labels
		},
		eventTrigger: {
			eventType,
			eventFilters,
			eventFilterPathPatterns,
			retry: opts.retry ?? false
		}
	};
}
function makeParams(event, service, connector, operation) {
	return {
		...service?.extractMatches(event.service),
		...connector?.extractMatches(event.connector),
		...operation?.extractMatches(event.operation)
	};
}
function onOperation(eventType, mutationOrOpts, handler) {
	const { service, connector, operation, opts } = getOpts(mutationOrOpts);
	const servicePattern = service ? new require_common_utilities_path_pattern.PathPattern(service) : undefined;
	const connectorPattern = connector ? new require_common_utilities_path_pattern.PathPattern(connector) : undefined;
	const operationPattern = operation ? new require_common_utilities_path_pattern.PathPattern(operation) : undefined;
	const func = (raw) => {
		const event = raw;
		const params = makeParams(event, servicePattern, connectorPattern, operationPattern);
		const { authtype, authid, service: service$1, connector: connector$1, operation: operation$1,...rest } = event;
		const dataConnectEvent = {
			...rest,
			authType: authtype,
			authId: authid,
			params
		};
		return require_v2_trace.wrapTraceContext(require_common_onInit.withInit(handler))(dataConnectEvent);
	};
	func.run = handler;
	func.__endpoint = makeEndpoint(eventType, opts, servicePattern, connectorPattern, operationPattern);
	return func;
}

//#endregion
Object.defineProperty(exports, 'dataconnect_exports', {
  enumerable: true,
  get: function () {
    return dataconnect_exports;
  }
});
exports.mutationExecutedEventType = mutationExecutedEventType;
exports.onMutationExecuted = onMutationExecuted;