const require_rolldown_runtime = require('../../_virtual/rolldown_runtime.js');
const require_runtime_manifest = require('../../runtime/manifest.js');
const require_common_encoding = require('../../common/encoding.js');
const require_common_onInit = require('../../common/onInit.js');
const require_v2_trace = require('../trace.js');
const require_v2_options = require('../options.js');

//#region src/v2/providers/pubsub.ts
var pubsub_exports = /* @__PURE__ */ require_rolldown_runtime.__export({
	Message: () => Message,
	onMessagePublished: () => onMessagePublished
});
/**
* Google Cloud Pub/Sub is a globally distributed message bus that automatically scales as you need it.
* You can create a function ({@link onMessagePublished}) that handles pub/sub events by using functions.pubsub.
*
* This function triggers whenever a new pub/sub message is sent to a specific topic.
* You must specify the Pub/Sub topic name that you want to trigger your function, and set the event within the
* onPublish() event handler.
*
* PubSub Topic:
* <ul>
*   <li>A resource that you can publish messages to and then consume those messages via subscriptions.
*   <li>An isolated data stream for pub/sub messages.
*   <li>Messages are published to a topic.
*   <li>Messages are listened to via a subscription.
*   <li>Each subscription listens to the messages published to exactly one topic.
*
* Subscriptions - Resource that listens to the messages published by exactly one topic.
*
* [More info here](https://firebase.google.com/docs/functions/pubsub-events)
*/
/**
* Interface representing a Google Cloud Pub/Sub message.
*
* @param data - Payload of a Pub/Sub message.
* @typeParam T - Type representing `Message.data`'s JSON format
*/
var Message = class {
	/**
	* @hidden
	* @alpha
	*/
	constructor(data) {
		this.messageId = data.messageId;
		this.data = data.data;
		this.attributes = data.attributes || {};
		this.orderingKey = data.orderingKey || "";
		this.publishTime = data.publishTime || new Date().toISOString();
		this._json = data.json;
	}
	/**
	* The JSON data payload of this message object, if any.
	*/
	get json() {
		if (typeof this._json === "undefined") {
			try {
				this._json = JSON.parse(Buffer.from(this.data, "base64").toString("utf8"));
			} catch (err) {
				throw new Error(`Unable to parse Pub/Sub message data as JSON: ${err.message}`);
			}
		}
		return this._json;
	}
	/**
	* Returns a JSON-serializable representation of this object.
	*
	* @returns A JSON-serializable representation of this object.
	*/
	toJSON() {
		const json = {
			messageId: this.messageId,
			data: this.data,
			publishTime: this.publishTime
		};
		if (Object.keys(this.attributes).length) {
			json.attributes = this.attributes;
		}
		if (this.orderingKey) {
			json.orderingKey = this.orderingKey;
		}
		return json;
	}
};
/**
* Handle a message being published to a Pub/Sub topic.
* @param topicOrOptions - A string representing the PubSub topic or an option (which contains the topic)
* @param handler - runs every time a Cloud Pub/Sub message is published
* @typeParam T - Type representing `Message.data`'s JSON format
*/
function onMessagePublished(topicOrOptions, handler) {
	let topic;
	let opts;
	if (typeof topicOrOptions === "string") {
		topic = topicOrOptions;
		opts = {};
	} else {
		topic = topicOrOptions.topic;
		opts = { ...topicOrOptions };
		delete opts.topic;
	}
	const func = (raw) => {
		const messagePublishedData = raw.data;
		messagePublishedData.message = new Message(messagePublishedData.message);
		return require_v2_trace.wrapTraceContext(require_common_onInit.withInit(handler))(raw);
	};
	func.run = handler;
	Object.defineProperty(func, "__trigger", { get: () => {
		const baseOpts$1 = require_v2_options.optionsToTriggerAnnotations(require_v2_options.getGlobalOptions());
		const specificOpts$1 = require_v2_options.optionsToTriggerAnnotations(opts);
		return {
			platform: "gcfv2",
			...baseOpts$1,
			...specificOpts$1,
			labels: {
				...baseOpts$1?.labels,
				...specificOpts$1?.labels
			},
			eventTrigger: {
				eventType: "google.cloud.pubsub.topic.v1.messagePublished",
				resource: `projects/${process.env.GCLOUD_PROJECT}/topics/${topic}`
			}
		};
	} });
	const baseOpts = require_v2_options.optionsToEndpoint(require_v2_options.getGlobalOptions());
	const specificOpts = require_v2_options.optionsToEndpoint(opts);
	const endpoint = {
		...require_runtime_manifest.initV2Endpoint(require_v2_options.getGlobalOptions(), opts),
		platform: "gcfv2",
		...baseOpts,
		...specificOpts,
		labels: {
			...baseOpts?.labels,
			...specificOpts?.labels
		},
		eventTrigger: {
			eventType: "google.cloud.pubsub.topic.v1.messagePublished",
			eventFilters: { topic },
			retry: opts.retry ?? false
		}
	};
	require_common_encoding.copyIfPresent(endpoint.eventTrigger, opts, "retry", "retry");
	func.__endpoint = endpoint;
	return func;
}

//#endregion
exports.Message = Message;
exports.onMessagePublished = onMessagePublished;
Object.defineProperty(exports, 'pubsub_exports', {
  enumerable: true,
  get: function () {
    return pubsub_exports;
  }
});